﻿Imports System.Drawing.Text
Imports Microsoft.Win32
Imports System.Security.Permissions

Public Class FormMain
    Private consoleFontsKey As RegistryKey

    Private Sub FormMain_Load(sender As Object, e As EventArgs) Handles MyBase.Load
        btnAdd.Enabled = False
        btnRemove.Enabled = False

        consoleFontsKey = Registry.LocalMachine.OpenSubKey("SOFTWARE\Microsoft\Windows NT\CurrentVersion\Console\TrueTypeFont", IsUserAdmin())

        SetSampleText(New Font(Me.Font.FontFamily, 18, FontStyle.Regular, GraphicsUnit.Pixel))
        LoadFonts()

        AddHandler CbMonospace.CheckedChanged, Sub() LoadFonts()
        AddHandler TbSampleSize.ValueChanged, Sub() UpdateSample()
    End Sub

    Private Sub SetSampleText(font As Font)
        Static labelsCreated As Boolean
        Static isBusy As Boolean
        If isBusy Then Exit Sub
        isBusy = True

        If Not labelsCreated Then TbSampleSize.Value = font.Size

        Me.SuspendLayout()

        Dim lbl As Label
        Dim p As New Point(LvAvailableFonts.Left, TbSampleSize.Top)
        Dim name As String
        Dim size = New Size(font.Size + 2, font.Size * 1.5)

        For i As Integer = 33 To 255
            name = "K" + i.ToString()

            If labelsCreated Then
                lbl = Me.Controls.Find(name, False).Single()
            Else
                lbl = New Label()
                Me.Controls.Add(lbl)

                lbl.BackColor = Color.White
                lbl.AutoSize = False
                lbl.BorderStyle = BorderStyle.FixedSingle
                lbl.TextAlign = ContentAlignment.MiddleCenter
                lbl.Name = name
            End If
            
            lbl.Size = size
            lbl.Location = p
            lbl.Text = Chr(i)
            lbl.Font = font

            p.X += lbl.Width - 1
            If p.X + lbl.Width >= TbSampleSize.Left Then
                p.X = LvAvailableFonts.Left
                p.Y += lbl.Height - 1
            End If
        Next

        Me.ResumeLayout()

        isBusy = False
        labelsCreated = True
    End Sub

    Private Function IsUserAdmin() As Boolean
        Return My.User.IsInRole(ApplicationServices.BuiltInRole.Administrator)
    End Function

    Private Sub LoadFonts()
        LoadAvailableFonts()
        LoadInstalledFonts()

        If LvAvailableFonts.Items.Count > 0 Then
            LvAvailableFonts.Items(0).Selected = True
            TbSampleSize.Enabled = True
        Else
            TbSampleSize.Enabled = False
        End If
    End Sub

    Private Sub LoadInstalledFonts()
        LvInstalledFonts.SuspendLayout()

        LvInstalledFonts.Items.Clear()

        If consoleFontsKey Is Nothing Then
            LvInstalledFonts.Enabled = False
        Else
            Dim names = consoleFontsKey.GetValueNames()
            Dim inactiveColor = Color.FromKnownColor(KnownColor.InactiveCaption)

            For Each item As ListViewItem In LvAvailableFonts.Items
                item.Selected = False
                item.ForeColor = Me.ForeColor
                item.Tag = 0
            Next

            For i As Integer = 0 To consoleFontsKey.ValueCount - 1
                Dim fontName = consoleFontsKey.GetValue(names(i)).ToString()
                Dim availableItem As ListViewItem = LvAvailableFonts.FindItemWithText(fontName)
                Dim newItem As ListViewItem = LvInstalledFonts.Items.Add(fontName)

                If availableItem IsNot Nothing Then
                    newItem.Font = availableItem.Font
                    availableItem.ForeColor = inactiveColor
                    availableItem.Tag = -1
                End If

                If names(i).StartsWith("0000") AndAlso Integer.Parse(names(i)) = 0 Then
                    newItem.Tag = names(i)
                Else
                    newItem.ForeColor = inactiveColor
                    newItem.Tag = -1
                End If
            Next

            LvInstalledFonts.AutoResizeColumns(ColumnHeaderAutoResizeStyle.ColumnContent)
        End If

        LvInstalledFonts.ResumeLayout()
    End Sub

    Private Sub LoadAvailableFonts()
        Dim fonts As New InstalledFontCollection()

        LvAvailableFonts.Items.Clear()

        Using g = Me.CreateGraphics()
            For Each family In fonts.Families
                Dim style = FontStyle.Regular

                If Not family.IsStyleAvailable(FontStyle.Regular) Then
                    For Each style In [Enum].GetValues(GetType(Drawing.FontStyle))
                        If family.IsStyleAvailable(style) Then Exit For
                    Next
                End If

                Dim f As New Font(family.Name, 11, style, GraphicsUnit.Point)

                ' This method doesn't work
                'Dim lf As New FontAPI.LOGFONT()
                'f.ToLogFont(lf, g)
                'If (lf.lfPitchAndFamily And FontAPI.FontPitchAndFamily.FIXED_PITCH) = FontAPI.FontPitchAndFamily.FIXED_PITCH Then
                '    LvAvailableFonts.Items.Add(f.Name).Font = f
                'End If

                ' This method kind of works, but it produces false positives
                'If TextRenderer.MeasureText(" ", f).Width = TextRenderer.MeasureText("W", f).Width Then
                '    LvAvailableFonts.Items.Add(f.Name).Font = f
                'End If

                ' This one works!
                If (Not CbMonospace.Checked) OrElse (CbMonospace.Checked AndAlso (FontAPI.GetTextMetrics(g, f).tmPitchAndFamily And 1) = 0) Then
                    LvAvailableFonts.Items.Add(f.Name).Font = f
                End If
            Next
        End Using

        LvAvailableFonts.AutoResizeColumns(ColumnHeaderAutoResizeStyle.ColumnContent)
    End Sub

    Private Sub LvAvailableFonts_SelectedIndexChanged(sender As Object, e As EventArgs) Handles LvAvailableFonts.SelectedIndexChanged
        If IsUserAdmin() Then
            Dim canAdd As Boolean = LvInstalledFonts.Enabled AndAlso (LvAvailableFonts.SelectedItems.Count > 0)

            If canAdd Then
                For Each item As ListViewItem In LvAvailableFonts.SelectedItems
                    If item.Tag = -1 Then
                        canAdd = False
                        Exit For
                    End If
                Next
            End If

            btnAdd.Enabled = canAdd
        End If

        UpdateSample()
    End Sub

    Private Sub UpdateSample()
        If LvAvailableFonts.SelectedItems.Count > 0 Then
            Dim family = LvAvailableFonts.SelectedItems(0).Font.FontFamily
            SetSampleText(New Font(family, TbSampleSize.Value, GetSupportedStyle(family), GraphicsUnit.Pixel))
        End If
    End Sub

    Private Function GetSupportedStyle(family As FontFamily) As FontStyle
        Dim style = FontStyle.Regular

        If Not family.IsStyleAvailable(FontStyle.Regular) Then
            For Each style In [Enum].GetValues(GetType(Drawing.FontStyle))
                If family.IsStyleAvailable(style) Then Exit For
            Next
        End If

        Return style
    End Function

    Private Sub LvInstalledFonts_SelectedIndexChanged(sender As Object, e As EventArgs) Handles LvInstalledFonts.SelectedIndexChanged
        If IsUserAdmin() Then
            Dim canRemove As Boolean = True
            For Each item As ListViewItem In LvInstalledFonts.SelectedItems
                If item.Tag = -1 Then
                    canRemove = False
                    Exit For
                End If
            Next
            btnRemove.Enabled = canRemove
        End If
    End Sub

    Private Sub btnAdd_Click(sender As Object, e As EventArgs) Handles btnAdd.Click
        Dim newValue As String = "0000"
        For Each item As ListViewItem In LvAvailableFonts.SelectedItems
            Dim names = consoleFontsKey.GetValueNames()
            Do
                If names.Contains(newValue.ToString()) Then
                    newValue += "0"
                Else
                    Exit Do
                End If
            Loop

            consoleFontsKey.SetValue(newValue, item.Font.FontFamily.Name)
        Next

        ' Lazy...
        LoadInstalledFonts()
    End Sub

    Private Sub btnRemove_Click(sender As Object, e As EventArgs) Handles btnRemove.Click
        For Each item As ListViewItem In LvInstalledFonts.SelectedItems
            consoleFontsKey.DeleteValue(item.Tag.ToString())
        Next

        ' Lazy...
        LoadInstalledFonts()
    End Sub
End Class
