﻿Imports System.ComponentModel

Public Class SubtitlesBrowser
    Private Enum Objects
        Captions
        ScrollBar
    End Enum

    Private Enum CaptionSections
        None
        Left
        Body
        Right
    End Enum

    Private Structure CaptionRectangle
        Public Rectangle As Rectangle
        Public Caption As Caption

        Public Sub New(caption As Caption, rectangle As Rectangle)
            Me.Rectangle = rectangle
            Me.Caption = caption
        End Sub
    End Structure

    Private mCaptions As List(Of Caption) = New List(Of Caption)
    Private mPosition As TimeSpan = TimeSpan.Zero
    Private mZoomFactor As Double = 10
    Private mSelectedCaption As Caption
    Private mRippleEdits As Boolean

    Private xOffset As Integer = 0
    Private isLeftMouseButtonDown As Boolean
    Private cursorPos As Point
    Private overObject As Objects = Objects.Captions
    Private captionSection As CaptionSections = CaptionSections.None
    Private overCaption As Caption = Nothing
    Private beforeEditToTime As TimeSpan
    Private captionChanged As Boolean
    Private captionFont As Font = New Font(Me.Font.FontFamily, 10, FontStyle.Regular, GraphicsUnit.Pixel)
    Private captionsRectangles As List(Of CaptionRectangle) = New List(Of CaptionRectangle)

    Private isShiftDown As Boolean
    Private isCtrlDown As Boolean

    Public Event CaptionSelected(caption As Caption, changed As Boolean)
    Public Event PositionChanged(position As TimeSpan, caption As Caption)
    Public Event CaptionsChanged(captions As List(Of Caption))

    <Browsable(False)>
    Public Property Position() As TimeSpan
        Get
            Return mPosition
        End Get
        Set(value As TimeSpan)
            mPosition = value
            Me.Invalidate()
        End Set
    End Property

    <Browsable(False)>
    Public Property SelectedCaption As Caption
        Get
            Return mSelectedCaption
        End Get
        Set(value As Caption)
            If mSelectedCaption <> value Then
                mSelectedCaption = value
                Me.Invalidate()
            End If
        End Set
    End Property

    <Category("Appearance")>
    Public Property ZoomFactor As Single
        Get
            Return mZoomFactor
        End Get
        Set(value As Single)
            SetZoomFactor(value)
        End Set
    End Property

    <Category("Behavior")>
    Public Property RippleEdits As Boolean
        Get
            Return mRippleEdits
        End Get
        Set(value As Boolean)
            mRippleEdits = value
        End Set
    End Property

    Public Sub CenterPosition(force As Boolean)
        Dim x As Integer = TimeToX(mPosition)
        If x < 0 OrElse x > Me.Width OrElse force Then
            Dim newOffset As Integer = mPosition.TotalSeconds * mZoomFactor - Me.Width \ 2
            If newOffset <> xOffset Then
                xOffset = newOffset
                CheckOffset()
                Me.Invalidate()
            End If
        End If
    End Sub

    Public Sub SetCaptions(captions As List(Of Caption), resetOffset As Boolean)
        mCaptions = captions
        If resetOffset Then xOffset = 0
        Me.Invalidate()
    End Sub

    Public Function GetCaptionAtPosition() As Caption
        Dim p As Point = New Point(TimeToX(mPosition), 3)
        For Each cr In captionsRectangles
            If cr.Rectangle.Contains(p) Then
                Return cr.Caption
            End If
        Next

        Return Nothing
    End Function

    Public Function TimeToX(time As TimeSpan) As Integer
        Return time.TotalSeconds * mZoomFactor - xOffset
    End Function

    Public Function XToTime(x As Integer) As TimeSpan
        Return TimeSpan.FromSeconds((x + xOffset) / mZoomFactor)
    End Function

    Private Sub SubtitlesBrowser_DoubleClick(sender As Object, e As System.EventArgs) Handles Me.DoubleClick
        If overObject = Objects.Captions Then
            If captionSection = CaptionSections.None Then
                If isShiftDown Then
                    Dim offset As Integer = xOffset
                    Dim referenceCaption As Caption = Nothing

                    For i As Integer = 0 To captionsRectangles.Count - 2
                        If captionsRectangles(i + 1).Rectangle.X > cursorPos.X Then
                            referenceCaption = captionsRectangles(i).Caption
                            Exit For
                        End If
                    Next

                    Dim newCaption As Caption
                    If referenceCaption Is Nothing Then
                        frmMain.lvOffsetted.SelectedItems.Clear()
                        newCaption = frmMain.AddCaption(True)
                    Else
                        frmMain.UpdateListViewCaption(frmMain.lvOffsetted, referenceCaption, True)
                        newCaption = frmMain.AddCaption()
                    End If

                    newCaption.FromTimeOffsetted = XToTime(cursorPos.X)
                    newCaption.ToTimeOffsetted = newCaption.FromTimeOffsetted + TimeSpan.FromSeconds(3)
                    frmMain.UpdateListViewCaption(frmMain.lvOffsetted, newCaption, True)
                    frmMain.EditCaption(True)

                    Me.Invalidate()
                End If
            Else
                frmMain.EditCaption(True)
            End If
        End If
    End Sub

    Private Sub SubtitlesBrowser_KeyDown(sender As Object, e As System.Windows.Forms.KeyEventArgs) Handles Me.KeyDown
        isShiftDown = e.Shift
        isCtrlDown = e.Control
    End Sub

    Private Sub SubtitlesBrowser_KeyUp(sender As Object, e As System.Windows.Forms.KeyEventArgs) Handles Me.KeyUp
        isShiftDown = e.Shift
        isCtrlDown = e.Control
    End Sub

    Private Sub SubtitlesBrowser_Load(sender As System.Object, e As System.EventArgs) Handles MyBase.Load
        Me.SetStyle(ControlStyles.AllPaintingInWmPaint, True)
        Me.SetStyle(ControlStyles.OptimizedDoubleBuffer, True)
        Me.SetStyle(ControlStyles.ResizeRedraw, True)
        Me.SetStyle(ControlStyles.Selectable, True)
        Me.SetStyle(ControlStyles.UserPaint, True)
    End Sub

    Private Sub SubtitlesBrowser_MouseDown(sender As Object, e As System.Windows.Forms.MouseEventArgs) Handles Me.MouseDown
        If mCaptions.Count > 0 AndAlso e.Button = Windows.Forms.MouseButtons.Left Then
            If e.Y < Me.Height - 10 Then
                overObject = Objects.Captions
                RaiseEvent PositionChanged(XToTime(e.Location.X), overCaption)
            Else
                overObject = Objects.ScrollBar
            End If
            isLeftMouseButtonDown = True
            cursorPos = e.Location
            captionChanged = False
            If overCaption IsNot Nothing Then beforeEditToTime = overCaption.ToTimeOffsetted
        End If
    End Sub

    Private Sub SubtitlesBrowser_MouseMove(sender As Object, e As System.Windows.Forms.MouseEventArgs) Handles Me.MouseMove
        If isLeftMouseButtonDown Then
            Dim delta As Integer = (cursorPos.X - e.X)
            Dim factor As Double = Math.Min(mZoomFactor / 10, Math.Abs(delta))

            If isCtrlDown Then factor *= 10

            Select Case overObject
                Case Objects.Captions
                    captionChanged = captionChanged Or (delta <> 0)
                    Select Case captionSection
                        Case CaptionSections.None
                            xOffset += (cursorPos.X - e.X)
                        Case CaptionSections.Left
                            overCaption.FromTimeOffsetted -= TimeSpan.FromMilliseconds(delta * factor)
                        Case CaptionSections.Right
                            overCaption.ToTimeOffsetted -= TimeSpan.FromMilliseconds(delta * factor)
                        Case CaptionSections.Body
                            overCaption.FromTimeOffsetted -= TimeSpan.FromMilliseconds(delta * factor)
                            overCaption.ToTimeOffsetted -= TimeSpan.FromMilliseconds(delta * factor)
                    End Select
                Case Objects.ScrollBar
                    xOffset = mCaptions.Last().ToTimeOffsetted.TotalSeconds * mZoomFactor * e.X / Me.Width
            End Select

            cursorPos.X = e.X
            Me.Invalidate()
        Else
            Me.Cursor = Cursors.Default
            captionSection = CaptionSections.None
            overCaption = Nothing

            For Each cr In captionsRectangles
                If cr.Rectangle.Contains(e.Location) Then
                    If e.X <= cr.Rectangle.X + 4 Then
                        Me.Cursor = Cursors.PanWest
                        captionSection = CaptionSections.Left
                    ElseIf e.X >= cr.Rectangle.Right - 4 Then
                        Me.Cursor = Cursors.PanEast
                        captionSection = CaptionSections.Right
                    Else
                        Me.Cursor = Cursors.NoMoveHoriz
                        captionSection = CaptionSections.Body
                    End If

                    overCaption = cr.Caption
                    Exit For
                End If
            Next
        End If
    End Sub

    Private Sub SubtitlesBrowser_MouseUp(sender As Object, e As System.Windows.Forms.MouseEventArgs) Handles Me.MouseUp
        If e.Button = Windows.Forms.MouseButtons.Left Then
            isLeftMouseButtonDown = False
            Me.Cursor = Cursors.Default

            If overObject = Objects.Captions AndAlso overCaption IsNot Nothing Then
                If mRippleEdits Then UpdateCaptions(overCaption, overCaption.ToTimeOffsetted - beforeEditToTime)

                mSelectedCaption = overCaption
                RaiseEvent CaptionSelected(overCaption, captionChanged)
                Me.Invalidate()
            End If
        End If
    End Sub

    Private Sub UpdateCaptions(fromCaption As Caption, offset As TimeSpan)
        If offset <> TimeSpan.Zero Then
            Dim captions As List(Of Caption) = New List(Of Caption)
            captions.Add(fromCaption)

            For i As Integer = mCaptions.IndexOf(fromCaption) + 1 To mCaptions.Count - 1
                mCaptions(i).FromTimeOffsetted += offset
                mCaptions(i).ToTimeOffsetted += offset
                captions.Add(mCaptions(i))
            Next

            RaiseEvent CaptionsChanged(captions)
        End If
    End Sub

    Private Sub SubtitlesBrowser_MouseWheel(sender As Object, e As System.Windows.Forms.MouseEventArgs) Handles Me.MouseWheel
        SetZoomFactor(mZoomFactor + e.Delta / 100)
    End Sub

    Private Sub SetZoomFactor(value As Single)

        If value < 1 Then
            mZoomFactor = 1
        Else
            mZoomFactor = value

            Dim offsetTime = XToTime(Me.Width / 2)
            xOffset = Me.Width / 2
            xOffset = TimeToX(offsetTime)
        End If

        Me.Invalidate()
    End Sub

    Private Sub CheckOffset()
        If mCaptions.Count = 0 Then Exit Sub

        If xOffset < 0 Then xOffset = 0
        Dim lastX As Integer = TimeToX(mCaptions.Last().ToTimeOffsetted)
        If lastX < 0 Then xOffset = TimeToX(mCaptions.Last().ToTimeOffsetted) + xOffset
    End Sub

    Private Sub SubtitlesBrowser_Paint(sender As Object, e As System.Windows.Forms.PaintEventArgs) Handles Me.Paint
        If mCaptions.Count = 0 Then Exit Sub

        CheckOffset()

        Dim g As Graphics = e.Graphics
        Dim r As Rectangle = Me.DisplayRectangle
        r.Inflate(-1, -1)

        captionsRectangles.Clear()

        Dim x1 As Integer
        Dim x2 As Integer
        For Each c In mCaptions
            x1 = TimeToX(c.FromTimeOffsetted)
            x2 = TimeToX(c.ToTimeOffsetted)
            If x2 < 0 Then Continue For

            Dim cr As Rectangle = New Rectangle(x1, 1, x2 - x1, r.Height - 11)

            If c = mSelectedCaption Then
                g.FillRectangle(Brushes.White, cr)
                g.DrawRectangle(Pens.Gray, cr)
            Else
                g.FillRectangle(Brushes.LightYellow, cr)
                g.DrawRectangle(Pens.DarkGray, cr)
            End If

            g.DrawString(c.TextOffsetted, captionFont, Brushes.Black, cr)

            captionsRectangles.Add(New CaptionRectangle(c, cr))

            If x2 > r.Width Then Exit For
        Next

        x1 = TimeToX(mCaptions.Last().ToTimeOffsetted) + xOffset
        x2 = TimeToX(mPosition)

        Dim w As Integer = CInt(Me.Width / (x1 / Me.Width))

        g.FillRectangle(Brushes.Gray, r.X, r.Height - 7, r.Width, 8)
        g.FillRectangle(Brushes.LightGray,
                            CInt((Me.Width - w) * xOffset / x1),
                            r.Height - 7,
                            w,
                            8)
        Using p As Pen = New Pen(Color.LightBlue, 3)
            g.DrawLine(p, x2 - 1, r.Height - 7, x2 - 1, 0)
        End Using
    End Sub
End Class
